<?php
/**
 * Plugin Name: 日夜模式切换器
 * Plugin URI: https://www.edzbe.com
 * Description: 为您的WordPress站点添加日夜模式切换功能，默认根据当地时间判断，并支持手动切换。
 * Version: 1.0.7
 * Author: 耳朵的主人
 * Author URI: https://www.edzbe.com
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: daynight-switcher
 * Domain Path: /languages
 */
// 阻止直接访问文件，增强安全性
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * 提前判断初始模式（PHP服务器端）
 * 优先级：Cookie (用户偏好) > 服务器时间
 * @return string 'light' 或 'dark'
 */
function dns_get_initial_mode_server_side() {
    // 1. 尝试从Cookie获取用户偏好
    if ( isset( $_COOKIE['theme-mode'] ) && in_array( $_COOKIE['theme-mode'], ['light', 'dark'] ) ) {
        return $_COOKIE['theme-mode'];
    }

    // 2. 如果Cookie不存在，根据服务器时间判断
    $hour = (int) current_time( 'H', true ); // 使用WordPress的 current_time 获取当前小时，避免时区问题
    return ($hour >= 6 && $hour < 18) ? 'light' : 'dark'; // 6-18点白天
}

/**
 * 注册并加载日夜模式脚本
 * 注意：日夜模式的样式文件不再在这里直接 enqueue，而是由 dns_output_conditional_styles 动态输出。
 */
function dns_enqueue_scripts() {
    // 获取主题目录URI
    $theme_css_dir_uri = get_template_directory_uri() . '/css/'; // 仍然传递给JS，以防需要手动构建 link 标签 fallback

    // 注册切换脚本（在页脚加载，确保DOM就绪）
    wp_enqueue_script(
        'daynight-switcher-script',
        plugins_url( 'js/daynight-switcher.js', __FILE__ ),
        array(),
        '1.0.0',
        true
    );

    // 将PHP数据传递给JS（如样式表ID、按钮ID、CSS文件路径）
    wp_localize_script(
        'daynight-switcher-script',
        'dns_data',
        array(
            'lightStyleId'      => 'daynight-switcher-light-style-css',
            'darkStyleId'       => 'daynight-switcher-dark-style-css',
            'buttonId'          => 'daynight-toggle',
         //   'textLight'         => esc_html__( '切换到暗夜模式', 'daynight-switcher' ),
         //   'textDark'          => esc_html__( '切换到白天模式', 'daynight-switcher' ),
            'lightStyleUrl'     => $theme_css_dir_uri . 'style-light.css', // 传递CSS路径给JS，用于动态创建link标签
            'darkStyleUrl'      => $theme_css_dir_uri . 'style-dark.css',  // 传递CSS路径给JS，用于动态创建link标签
        )
    );
}
add_action( 'wp_enqueue_scripts', 'dns_enqueue_scripts' ); // 重命名函数，因为不再enqueue styles

/**
 * 在页面<body>开头添加切换按钮（兼容WP5.2+）
 */
function dns_add_toggle_button() {
    if ( is_admin() ) return; // 排除后台
    ?>
    <button id="daynight-toggle" class="daynight-toggle-button" aria-label="<?php esc_attr_e( '切换日夜模式', 'daynight-switcher' ); ?>">
        <?php esc_html_e( '加载中...', 'daynight-switcher' ); ?>
    </button>
    <?php
}
add_action( 'wp_body_open', 'dns_add_toggle_button' );

/**
 * 核心：在页面头部根据PHP判断动态输出正确的样式表和body类
 * 这将避免任何模式切换时的闪烁。
 */
function dns_output_conditional_styles() {
    if ( is_admin() ) return; // 排除后台

    $initial_mode = dns_get_initial_mode_server_side();
    $theme_css_dir_uri = get_template_directory_uri() . '/css/';
    $version = '1.0.0'; // 样式表的版本号

    // 输出正确的样式表
    if ( $initial_mode === 'light' ) {
        echo '<link rel="stylesheet" id="daynight-switcher-light-style-css" href="' . esc_url( $theme_css_dir_uri . 'style-light.css?ver=' . $version ) . '" type="text/css" media="all">';
        echo '<link rel="stylesheet" id="daynight-switcher-dark-style-css" href="' . esc_url( $theme_css_dir_uri . 'style-dark.css?ver=' . $version ) . '" type="text/css" media="not all">';
    } else { // dark mode
        echo '<link rel="stylesheet" id="daynight-switcher-light-style-css" href="' . esc_url( $theme_css_dir_uri . 'style-light.css?ver=' . $version ) . '" type="text/css" media="not all">';
        echo '<link rel="stylesheet" id="daynight-switcher-dark-style-css" href="' . esc_url( $theme_css_dir_uri . 'style-dark.css?ver=' . $version ) . '" type="text/css" media="all">';
    }

    // 立即在body标签上设置正确的class，防止任何延迟闪烁
    // 注意：这个内联脚本在<?php标签内，所以它会先于任何其他JS执行
    ?>
    <script>
        // 确保body class在HTML解析时就存在
        document.documentElement.classList.add('<?php echo esc_js( $initial_mode ); ?>-mode');
        document.body.classList.add('<?php echo esc_js( $initial_mode ); ?>-mode');

        // 后备或纠正逻辑（如果localStorage与PHP判断不符，或Cookie过期）
        (function() {
            const savedMode = localStorage.getItem('theme-mode');
            const currentInitialMode = '<?php echo esc_js( $initial_mode ); ?>'; // PHP决定的初始模式

            if (savedMode && savedMode !== currentInitialMode) {
                // 如果localStorage和PHP的初始判断不一致，JS会立即纠正
                const lightStyle = document.getElementById('daynight-switcher-light-style-css');
                const darkStyle = document.getElementById('daynight-switcher-dark-style-css');

                if (lightStyle && darkStyle) {
                    if (savedMode === 'light') {
                        lightStyle.media = 'all';
                        darkStyle.media = 'not all';
                    } else { // savedMode === 'dark'
                        lightStyle.media = 'not all';
                        darkStyle.media = 'all';
                    }
                }
                document.documentElement.classList.remove('light-mode', 'dark-mode');
                document.documentElement.classList.add(`${savedMode}-mode`);
                document.body.classList.remove('light-mode', 'dark-mode');
                document.body.classList.add(`${savedMode}-mode`);
            }
        })();
    </script>
    <?php
}
// 在<head>标签内优先执行（优先级1，确保最早运行）
add_action( 'wp_head', 'dns_output_conditional_styles', 1 );

