document.addEventListener('DOMContentLoaded', function() {
    // 检查本地化数据是否正常
    if (typeof dns_data === 'undefined') {
        console.error('日夜模式切换器：未获取到本地化数据，请检查PHP的wp_localize_script调用！');
        return;
    }

    // 解构本地化数据
    const { lightStyleId, darkStyleId, buttonId, textLight, textDark, lightStyleUrl, darkStyleUrl } = dns_data;

    // 获取DOM元素
    const lightStyle = document.getElementById(lightStyleId);
    const darkStyle = document.getElementById(darkStyleId);
    const toggleButton = document.getElementById(buttonId);

    // 确保样式表元素存在。如果PHP动态输出失败，这里可能是null。
    // 但是在最新的方案中，PHP一定会输出这两个link标签，只是media属性不同。
    if (!lightStyle || !darkStyle || !toggleButton) {
        // Fallback: 如果PHP没有输出link标签，JS动态创建它们
        console.warn('日夜模式切换器：未找到核心样式表元素。尝试动态创建。');
        // 如果lightStyle或darkStyle不存在，这里需要动态创建。
        // 为了简化，这里假设它们总是由PHP输出。如果真的未找到，需要更复杂的JS fallback。
        // 目前先跳过，因为PHP应该确保它们存在。
        // return; // 暂时不返回，让下面的逻辑有机会运行
    }


    console.log('日夜模式切换器：核心元素加载完成！');

    /**
     * 设置Cookie
     * @param {string} name - Cookie名称
     * @param {string} value - Cookie值
     * @param {number} days - Cookie过期天数
     */
    function setCookie(name, value, days) {
        let expires = "";
        if (days) {
            const date = new Date();
            date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
            expires = "; expires=" + date.toUTCString();
        }
        document.cookie = name + "=" + (value || "")  + expires + "; path=/";
    }

    /**
     * 设置主题模式（更新样式、按钮文本、body类、同步Cookie）
     * @param {string} mode - 'light'或'dark'
     * @param {boolean} save - 是否保存到localStorage和Cookie
     */
    function setMode(mode, save = true) {
        // 更新样式表状态，通过修改 media 属性来激活/禁用样式
        if (mode === 'light') {
            if (lightStyle) lightStyle.media = 'all';     // 激活白天样式
            if (darkStyle) darkStyle.media = 'not all';  // 禁用暗夜样式
            toggleButton.textContent = textLight;
            toggleButton.setAttribute('aria-pressed', 'false');
        } else { // mode === 'dark'
            if (lightStyle) lightStyle.media = 'not all'; // 禁用白天样式
            if (darkStyle) darkStyle.media = 'all';      // 激活暗夜样式
            toggleButton.textContent = textDark;
            toggleButton.setAttribute('aria-pressed', 'true');
        }

        // 更新body和html根元素的类（方便扩展CSS）
        document.documentElement.classList.remove('light-mode', 'dark-mode');
        document.documentElement.classList.add(`${mode}-mode`);
        document.body.classList.remove('light-mode', 'dark-mode');
        document.body.classList.add(`${mode}-mode`);

        // 保存用户偏好（可选）到 localStorage 和 Cookie
        if (save) {
            localStorage.setItem('theme-mode', mode);
            setCookie('theme-mode', mode, 30); // Cookie保存30天
        }
    }

    /**
     * 获取系统默认模式
     * @returns {string} 'light'或'dark'
     */
    function getSystemMode() {
        const hour = new Date().getHours();
        return (hour >= 6 && hour < 18) ? 'light' : 'dark';
    }

    // --- 初始化 ---
    // 页面加载时的初始模式判断和样式设置已由PHP在服务器端完成，
    // 这里的JS初始化主要是确保按钮文本正确，并处理LocalStorage和Cookie可能不同步的情况。
    const savedMode = localStorage.getItem('theme-mode');
    let initialMode;

    if (savedMode && (savedMode === 'light' || savedMode === 'dark')) {
        initialMode = savedMode;
    } else {
        initialMode = getSystemMode();
    }

    // 确保当前的Cookie也与initialMode一致，防止PHP下次加载时判断错误
    setCookie('theme-mode', initialMode, 30);

    // 调用 setMode 来同步按钮文本，以及确保 body class 即使在PHP未能设置时也能被设置（虽然PHP现在会设置）
    setMode(initialMode, false); 
    console.log(`日夜模式切换器：初始模式为${initialMode}`);


    // --- 切换按钮点击事件 ---
    toggleButton.addEventListener('click', function() {
        // 从html根元素获取当前模式，因为CSS可能也通过html根元素控制
        const currentMode = document.documentElement.classList.contains('light-mode') ? 'light' : 'dark';
        const newMode = currentMode === 'light' ? 'dark' : 'light';
        setMode(newMode); // 切换并保存偏好
        console.log(`日夜模式切换器：已切换至${newMode}`);
    });
});
